definition module StdWindow


//	********************************************************************************
//	Clean Standard Object I/O library, version 1.2
//	
//	StdWindow defines functions on windows and dialogues.
//	********************************************************************************


from	StdFunc			import St
import	StdControlClass, StdWindowDef
from	StdPSt			import PSt, IOSt


//	Functions applied to non-existent windows or unknown ids have no effect.
class Windows wdef where
	openWindow		:: .ls !(wdef .ls (PSt .l)) !(PSt .l)
								-> (!ErrorReport,!PSt .l)
	getWindowType	::      (wdef .ls .pst)	-> WindowType

class Dialogs wdef where
	openDialog		:: .ls !(wdef .ls (PSt .l))    !(PSt .l)
					->  (  !ErrorReport,            !PSt .l)
	openModalDialog	:: .ls !(wdef .ls (PSt .l))    !(PSt .l)
					->  (!(!ErrorReport,!Maybe .ls),!PSt .l)
	getDialogType	::      (wdef .ls .pst)	-> WindowType

/*	open(Window/Dialog) opens the given window(dialog).
	If the Window(Dialog) has no WindowIndex attribute (see StdWindowDef), then the 
	new window is opened frontmost. 
	If the Window(Dialog) has a WindowIndex attribute, then the new window is 
	opened behind the window indicated by the integer index:
		Index value 1 indicates the top-most window. 
		Index value M indicates the bottom-most modal window, if there are M modal 
			windows. 
		Index value N indicates the bottom-most window, if there are N windows. 
		If index<M, then the new window is added behind the bottom-most modal window
			(at index M). 
		If index>N, then the new window is added behind the bottom-most window
			(at index N). 
	openModalDialog always opens a window at the front-most position. 
	openWindow may not be permitted to open a window depending on its 
		DocumentInterface (see the comments at the ShareProcesses instances in 
		module StdProcess).
	In case the window does not have an Id, it will obtain a fresh Id. The Id can 
	be reused after closing this window. 
	In case a window with the same Id is already open the window will not be opened. 
	In case controls are opened with duplicate Ids, the window will not be opened.
	openModalDialog terminates when:
		the modal dialog has been closed (by means of closeWindow), or the process 
		has been terminated (by means of closeProcess). If the ErrorReport==NoError,
		then also the final local state of the modal dialog is returned, otherwise
		Nothing.
*/

instance Windows (Window c)	| Controls c
instance Dialogs (Dialog c) | Controls c


closeWindow			:: !Id !(PSt .l) -> PSt .l
closeActiveWindow	::     !(PSt .l) -> PSt .l
/*	If the indicated window is not an inactive modal dialog, then closeWindow closes 
		the window/dialogue.
		In case the Id of the window was generated by open(Window/Dialog), it will 
		become reusable for new windows/dialogues.
		No window is closed in case of an unknown Id.
	closeActiveWindow closes the currently active window/dialogue (see also
		getActiveWindow) if such a window could be found.
*/


setActiveWindow	:: !Id		!( PSt .l) -> PSt .l
getActiveWindow	::			!(IOSt .l) -> (!Maybe Id,!IOSt .l)
/*	setActiveWindow makes the indicated window the active window. 
		If there are modal dialogues, then the window will be placed behind the last
		modal dialog.
		setActiveWindow has no effect in case the window is unknown or is a modal 
		dialog.
	getActiveWindow returns the Id of the window that currently has the input focus
		of the interactive process. 
		Nothing is returned if there is no such window.
*/

setActiveControl:: !Id		!( PSt .l) -> PSt .l
getActiveControl::			!(IOSt .l) -> (!(!Bool,!Maybe Id),!IOSt .l)
/*	setActiveControl makes the indicated (PopUp/Edit/Custom/Compound)Control the 
		active control. This succeeds only if its parent window is already active.
	getActiveControl returns the Id of the (PopUp/Edit/Custom/Compound)Control that 
		currently has the input focus.
		The Boolean result is True only iff such a control could be found.
		Nothing is returned if the control has no Id attribute or if the Boolean 
		result is False. 
*/


stackWindow		:: !Id !Id	!(PSt .l) -> PSt .l
/*	stackWindow id1 id2 places the window with id1 behind the window with id2. 
	If id1 or id2 is unknown, or id1 indicates a modal window, stackWindow does 
	nothing.
	If id2 indicates a modal window, then the window with id1 is placed behind the 
	last modal window. 
*/

getWindowStack	:: !(IOSt .l) -> (![(Id,WindowType)],	!IOSt .l)
getWindowsStack	:: !(IOSt .l) -> (![Id],				!IOSt .l)
getDialogsStack	:: !(IOSt .l) -> (![Id],				!IOSt .l)
/*	getWindowStack returns the Ids and WindowTypes of all currently open windows, 
	in the current stacking order starting with the active window.
	get(Windows/Dialogs)Stack is equal to getWindowStack, restricted to Windows 
	instances and Dialogs instances respectively.
*/


getDefaultHMargin	:: !Bool	!(IOSt .l) -> ((Int,Int),		!IOSt .l)
getDefaultVMargin	:: !Bool	!(IOSt .l) -> ((Int,Int),		!IOSt .l)
getDefaultItemSpace	:: !Bool	!(IOSt .l) -> ((Int,Int),		!IOSt .l)
getWindowHMargin	:: !Id		!(IOSt .l) -> (!Maybe (Int,Int),!IOSt .l)
getWindowVMargin	:: !Id		!(IOSt .l) -> (!Maybe (Int,Int),!IOSt .l)
getWindowItemSpace	:: !Id		!(IOSt .l) -> (!Maybe (Int,Int),!IOSt .l)
/*	getDefault((H/V)Margin)/ItemSpace) isWindow return the default values for the 
		horizontal and vertical window (if isWindow)/dialogue (if (not isWindow)) 
		margins and item spaces.
	getWindow((H/V)Margin/ItemSpace) return the current horizontal and vertical 
		margins and item spaces of the indicated window. These will have the default 
		values in case they were not specified as an attribute.
		In case the window does not exist, Nothing is yielded.
*/


enableWindow			:: !Id	!(IOSt .l) -> IOSt .l
disableWindow			:: !Id	!(IOSt .l) -> IOSt .l
enableWindowMouse		:: !Id	!(IOSt .l) -> IOSt .l
disableWindowMouse		:: !Id	!(IOSt .l) -> IOSt .l
enableWindowKeyboard	:: !Id	!(IOSt .l) -> IOSt .l
disableWindowKeyboard	:: !Id	!(IOSt .l) -> IOSt .l
/*	(en/dis)ableWindow
		(en/dis)ables the indicated window.
	(en/dis)ableWindowMouse
		(en/dis)ables mouse handling of the indicated window.
	(en/dis)ableWindowKeyboard
		(en/dis)ables keyboard handling of the indicated window.
	Disabling a window overrules the SelectStates of its elements, which all become 
	Unable. 
	Reenabling the window reestablishes the SelectStates of its elements.
	The functions have no effect in case of invalid Ids or Dialogs instances.
	The latter four functions also have no effect in case the Window does not have 
	the indicated attribute.
*/


getWindowSelectState		:: !Id	!(IOSt .l) ->(!Maybe SelectState,!IOSt .l)
getWindowMouseSelectState	:: !Id	!(IOSt .l) ->(!Maybe SelectState,!IOSt .l)
getWindowKeyboardSelectState:: !Id	!(IOSt .l) ->(!Maybe SelectState,!IOSt .l)
/*	getWindowSelectState
		yields the current SelectState of the indicated window.
	getWindow(Mouse/Keyboard)SelectState
		yields the current SelectState of the mouse/keyboard of the indicated 
		window.
	The functions return Nothing in case of invalid Ids or Dialogs instances or if 
	the Window does not have the indicated attribute.
*/


getWindowMouseStateFilter	:: !Id							!(IOSt .l)
							-> (!Maybe MouseStateFilter,	! IOSt .l)
getWindowKeyboardStateFilter:: !Id							!(IOSt .l)
							-> (!Maybe KeyboardStateFilter,	! IOSt .l)
setWindowMouseStateFilter	:: !Id !MouseStateFilter		!(IOSt .l)
														   -> IOSt .l
setWindowKeyboardStateFilter:: !Id !KeyboardStateFilter		!(IOSt .l)
														   -> IOSt .l
/*	getWindow(Mouse/Keyboard)StateFilter yields the current 
		(Mouse/Keyboard)StateFilter of the indicated window. Nothing is yielded in 
		case the window does not exist or has no Window(Mouse/Keyboard) attribute.
	setWindow(Mouse/Keyboard)StateFilter replaces the current 
		(Mouse/Keyboard)StateFilter of the indicated window. If the indicated window
		does not exist the function has no effect.
*/


appWindowPicture:: !Id !.(IdFun *Picture) !(IOSt .l) -> IOSt .l
accWindowPicture:: !Id !.(St *Picture .x) !(IOSt .l) -> (!Maybe .x,!IOSt .l)
/*	(app/acc)WindowPicture applies the given drawing function to the Picture of
	the indicated window (behind all controls). 
	Both functions have no effect in case the window is unknown or is a Dialog.
	In that case, accWindowPicture also returns Nothing.
*/


updateWindow	:: !Id !(Maybe ViewFrame)	!(IOSt .l) -> IOSt .l
/*	updateWindow applies the WindowLook attribute function of the indicated window.
	The Look attribute function is applied to the following arguments:
	The current SelectState of the window, and
	the UpdateState argument
		{oldFrame=viewframe,newFrame=viewframe,updArea=[frame]}
	where viewframe is the current ViewFrame of the window;
	and frame depends on the optional ViewFrame argument:
		in case of (Just rectangle):
			the intersection of viewframe and rectangle.
		in case of Nothing:
			viewframe. 
	updateWindow has no effect in case of unknown windows, or if the indicated 
	window is a Dialog, or the optional viewframe argument is an empty rectangle.
*/

setWindowLook	:: !Id !Bool !(!Bool,!Look)	!(IOSt .l) -> IOSt .l
getWindowLook	:: !Id						!(IOSt .l)
					  -> (!Maybe (Bool,Look),!IOSt .l)
/*	setWindowLook sets the (render/look) of the indicated window. 
		The window is redrawn only if the Boolean argument is True. 
		setWindowLook has no effect in case the window does not exist, or is a 
		Dialog.
	getWindowLook returns the (Just (render/look)) of the indicated window. 
		In case the window does not exist, or is a Dialog, or has no WindowLook 
		attribute, the result is Nothing. 
*/


setWindowPos	:: !Id !ItemPos	!(IOSt .l) -> IOSt .l
getWindowPos	:: !Id			!(IOSt .l) -> (!Maybe Vector2,!IOSt .l)
/*	setWindowPos places the window at the indicated position. 
		If the ItemPos argument refers to the Id of an unknown window (in case of 
		LeftOf/RightTo/Above/Below), setWindowPos has no effect. 
		If the ItemPos argument is one of (LeftOf/RightTo/Above/Below)Prev, then the
		previous window is the window that is before the window in the current 
		stacking order. 
		If the window is frontmost, setWindowPos has no effect. setWindowPos also 
		has no effect if the window would be moved outside the screen, or if the Id
		is unknown or refers to a modal Dialog.
	getWindowPos returns the current item offset position of the indicated window.
		The corresponding ItemPos is (LeftTop,OffsetVector offset). Nothing is 
		returned in case the window does not exist.
*/


moveWindowViewFrame	:: !Id Vector2 !(IOSt .l) -> IOSt .l
/*	moveWindowViewFrame moves the orientation of the view frame of the indicated 
	window over the given vector, and updates the window if necessary. The view 
	frame is not moved outside the ViewDomain of the window. 
	In case of unknown Id, or of Dialogs, moveWindowViewFrame has no effect.
*/

getWindowViewFrame	:: !Id !(IOSt .l) -> (!ViewFrame,!IOSt .l)
/*	getWindowViewFrame returns the current view frame of the window in terms of the
	ViewDomain. Note that in case of a Dialog, getWindowViewFrame returns 
	{zero,size}.
	In case of unknown windows, the ViewFrame result is zero.
*/

setWindowViewSize	:: !Id !Size	!(IOSt .l) -> IOSt .l
getWindowViewSize	:: !Id			!(IOSt .l) -> (!Size,!IOSt .l)
/*	setWindowViewSize
		sets the size of the view frame of the indicated window as given, and 
		updates the window if necessary. The size is fit between the minimum size 
		and the screen dimensions. 
		In case of unknown Ids, or of Dialogs, setWindowViewSize has no effect.
	getWindowViewSize yields the current size of the view frame of the indicated 
		window. If the window does not exist, zero is returned.
*/

setWindowOuterSize	:: !Id !Size	!(IOSt .l) -> IOSt .l
getWindowOuterSize	:: !Id			!(IOSt .l) -> (!Size,!IOSt .l)
/*	setWindowOuterSize
		sets the size of the outer frame of the indicated window as given, and 
		updates the window if necessary. The size is fit between the minimum size 
		and the screen dimensions. 
		In case of unknown Ids, or of Dialogs, setWindowOuterSize has no effect.
	getWindowOuterSize yields the current size of the outer frame of the indicated 
		window. If the window does not exist, zero is returned.
*/

setWindowViewDomain	:: !Id ViewDomain	!(IOSt .l) -> IOSt .l
getWindowViewDomain	:: !Id				!(IOSt .l)
				   -> (!Maybe ViewDomain,!IOSt .l)
/*	setWindowViewDomain
		sets the view domain of the indicated window as given. The window view frame
		is moved such that a maximum portion of the view domain is visible. The 
		window is not resized.
		In case of unknown Ids, or of Dialogs, setWindowViewDomain has no effect.
	getWindowViewDomain
		returns the current ViewDomain of the indicated window. 
		Nothing is returned in case the window does not exist or is a Dialog.
*/

setWindowScrollFunction :: !Id Direction ScrollFunction !(IOSt .l) -> IOSt .l
getWindowScrollFunction :: !Id Direction                !(IOSt .l)
							   -> (!Maybe ScrollFunction,!IOSt .l)
/*	setWindowScrollFunction 
		changes the current scroll function of the indicated Window and direction
		only if the indicated window already had a scroll bar in that direction. 
		In all other cases setWindowScrollFunction has no effect.
	getWindowScrollFunction
		returns the current scroll function in the argument direction if the 
		indicated Window had one. 
		In all other cases Nothing is returned. 
*/


setWindowTitle	:: !Id Title       !(IOSt .l) -> IOSt .l
setWindowOk		:: !Id Id          !(IOSt .l) -> IOSt .l
setWindowCancel	:: !Id Id          !(IOSt .l) -> IOSt .l
setWindowCursor	:: !Id CursorShape !(IOSt .l) -> IOSt .l
getWindowTitle	:: !Id             !(IOSt .l) -> (!Maybe Title,      !IOSt .l)
getWindowOk		:: !Id             !(IOSt .l) -> (!Maybe Id,         !IOSt .l)
getWindowCancel	:: !Id             !(IOSt .l) -> (!Maybe Id,         !IOSt .l)
getWindowCursor	:: !Id             !(IOSt .l) -> (!Maybe CursorShape,!IOSt .l)
/*	setWindow(Title/Ok/Cancel/Cursor) set the indicated window attributes.
		In case of unknown Ids, these functions have no effect.
	getWindow(Title/Ok/Cancel/Cursor) get the indicated window attributes.
		In case of unknown Ids, the result is Nothing.
*/
